﻿"""
Transport which makes a discrete move to a pre-specified destination.
"""

import viz
import vizact
import vizmat

import transportation


# Translation and/or rotation modes
DISABLED = viz.OFF
MATCH_TRANSPORT = 1
MATCH_PIVOT = 2

# Travel modes
IMMEDIATE = 1
SPEED_BASED = 2
TIME_BASED = 3


class Teleport(transportation.Transport):
	"""Transport providing teleportation style movement.
	
	Moves in single operations to a target position and orientation.
	
	@param translationMode: 
	if DISABLED (or False), the transport will not move its position.
	
	if MATCH_TRANSPORT, the transport itself will travel to the target
	position.
	
	if MATCH_PIVOT, the transport will automatically adjust itself in
	such a way as that the pivot itself ends up at the target position,
	and the transport itself will end up wherever it needs to in order
	to make that happen. In other words the transport will "deliver"
	the pivot to the target position.
	
	@param rotationMode:
	if DISABLED (or False) transport will not move its orientation.
	
	if MATCH_TRANSPORT, the transport itself will travel to match the
	target orientation.
	
	if MATCH_PIVOT, the transport will automatically adjust itself in
	such a way as that the pivot itself ends up facing the target
	orientation, and the transport itself will end up facing whichever
	orientation it needs to in order to make that happen. In other
	words, the transport will "deliver" the pivot to the target
	orientation.
	
	@param travelMode:
	if IMMEDIATE, will move to destination instantly.
	
	if TIME_BASED, uses a fixed time duration to reach destination.
	
	if SPEED_BASED, will travel to destination at fixed speed.
	
	@param travelTime: desired travel time, in seconds, to transition
	to the target position and orientation.
	
	@param speed: desired travel speed, in meters per second, at
	which to transition to the target position.
	
	@param rotationalSpeed: desired rotational speed, in degrees per
	second, at which to transition to the target orientation.
	"""
	
	def __init__(self,
					translationMode=MATCH_PIVOT,
					rotationMode=DISABLED,
					travelMode=TIME_BASED,
					travelTime=2,
					speed=1.4,
					rotationalSpeed=45.0,
					**kwargs):
		
		# Init the base transport object
		transportation.Transport.__init__(self, **kwargs)
		
		# Store passed-in parameters
		self._translationMode = translationMode
		self._rotationMode = rotationMode
		self._travelMode = travelMode
		self._travelTime = travelTime
		self._speed = speed
		self._rotationalSpeed = rotationalSpeed
		
		# Internal variables for keeping track of destination
		self._targetPos = None
		self._targetEuler = None
	
	def createConfigUI(self):
		"""Creates the vizconfig configuration ui.
		
		You do not need to call this function directly.
		"""
		ui = vizconfig.DefaultUI()
		ui.addBoolItem('Enabled', self.updateEvent.setEnabled, self.updateEvent.getEnabled)
		ui.addFloatItem('Travel time', fset=self.setTravelTime, fget=self.getTravelTime)
		ui.addFloatItem('Speed', fset=self.setSpeed, fget=self.getSpeed)
		ui.addFloatItem('Rotational speed', fset=self.setRotationalSpeed, fget=self.getRotationalSpeed)
		ui.addChoiceListItem('Travel mode',
								choices=[('IMMEDIATE', IMMEDIATE),
										('SPEED_BASED', SPEED_BASED),
										('TIME_BASED', TIME_BASED)],
								fset=self.setTravelMode,
								fget=self.getTravelMode
								)
		movementModeChoices = [('DISABLED', DISABLED),
								('MATCH_TRANSPORT', MATCH_TRANSPORT),
								('MATCH_PIVOT', MATCH_PIVOT)
								]
		ui.addChoiceListItem('Translation mode',
								choices=movementModeChoices,
								fset=self.setTranslationMode,
								fget=self.getTranslationMode
								)
		ui.addChoiceListItem('Orientation mode',
								choices=movementModeChoices,
								fset=self.setRotationMode,
								fget=self.getRotationMode
								)
		return ui
	
	def setTranslationMode(self, mode):
		"""Sets the translation mode. See class docstring."""
		self._translationMode = mode
	
	def getTranslationMode(self):
		"""Returns the translation mode. See module constants."""
		return self._translationMode
	
	def setRotationMode(self, mode):
		"""Sets the rotation mode. See class docstring."""
		self._rotationMode = mode
	
	def getRotationMode(self):
		"""Returns the rotation mode. See module constants."""
		return self._rotationMode
	
	def setTravelMode(self, mode):
		"""Sets the travel mode. See class docstring."""
		self._travelMode = mode
	
	def getTravelMode(self):
		"""Returns the travel mode. See module constants."""
		return self._travelMode
	
	def setTravelTime(self, travelTime):
		"""Sets the travel duration in seconds"""
		self._travelTime = travelTime
	
	def getTravelTime(self):
		"""Returns the travel duration in seconds."""
		return self._travelTime
	
	def setSpeed(self, speed):
		"""Sets the translation speed in meters/second."""
		self._speed = speed
	
	def getSpeed(self):
		"""Returns the translation speed in meters/second"""
		return self._speed
	
	def setRotationalSpeed(self, rotationalSpeed):
		"""Sets the rotation speed in degrees/second"""
		self._rotationalSpeed = rotationalSpeed
	
	def getRotationalSpeed(self):
		"""Returns the rotation speed in degrees/second"""
		return self._rotationalSpeed
	
	def teleportTo(self, targetPos, targetEuler):
		"""Move to the target position and orientation.
		
		@param targetPos: a list of position coordinates [x, y, z]
		@param targetEuler: a list of Euler angles [yaw, pitch, roll]
		"""
		self._targetPos = targetPos
		self._targetEuler = targetEuler
		
		# If using a pivot, alter target destination to accommodate
		if self._pivot:
			self._alterTargetsForPivot()
		
		# Travel in one of the given modes
		if self._travelMode == IMMEDIATE:
			self._travelImmediately()
		else:
			self._travelGradually()
	
	def _alterTargetsForPivot(self):
		"""Alters the target position and orientation given a pivot.
		
		Target coordinates for the transport will be modified such
		that, when the transport reaches the (adjusted) target
		coordinates, the pivot will end up at the original target
		coordinates.
		"""
		# Get handles to the transport and pivot matrices
		transportMatrix = self.getMatrix(viz.ABS_GLOBAL)
		pivotMatrix = self._pivot.getMatrix(viz.ABS_GLOBAL)
		
		# Find the pivot's transformation relative to the transport's
		relativeMatrix = pivotMatrix * transportMatrix.inverse()
		relativePosition = vizmat.Vector(relativeMatrix.getPosition())
		
		# Create a placeholder matrix to store computations
		placeholderMatrix = vizmat.Transform()
		
		# Set up the placeholder as needed for the given rotation mode
		if self._rotationMode == MATCH_TRANSPORT:
			placeholderMatrix.setEuler(self._targetEuler)
		elif self._rotationMode == MATCH_PIVOT:
			placeholderMatrix.setEuler(self._targetEuler)
			placeholderMatrix = placeholderMatrix * relativeMatrix.inverse()
			placeholderMatrix.setPosition(0, 0, 0)
		else:
			placeholderMatrix.setQuat(transportMatrix.getQuat())
		
		# Alter the target euler given the placeholder
		self._targetEuler = placeholderMatrix.getEuler()
		
		# Alter the target position if needed
		if self._translationMode == MATCH_PIVOT:
			# Get the pivot position relative to the placeholder
			positionOffset = placeholderMatrix.preMultVec(-relativePosition)
			# Alter the target position accordingly
			self._targetPos = vizmat.Vector(positionOffset) + self._targetPos
	
	def _travelImmediately(self):
		"""Set the position and orientation to the targets"""
		if self._translationMode != DISABLED:
			self.setPosition(self._targetPos, viz.ABS_GLOBAL)
		if self._rotationMode != DISABLED:
			self.setEuler(self._targetEuler, viz.ABS_GLOBAL)
	
	def _travelGradually(self):
		"""Move and rotate to the targets via animation action"""
		# Set up keyword argument dictionaries for action functions
		kwargs = {'interpolate': vizact.easeInOut, 'mode': viz.ABS_GLOBAL}
		
		# Set up move and spin actions
		if self._travelMode == TIME_BASED:
			move = vizact.moveTo(self._targetPos, time=self._travelTime, **kwargs)
			spin = vizact.spinTo(euler=self._targetEuler, time=self._travelTime, **kwargs)
		if self._travelMode == SPEED_BASED:
			move = vizact.moveTo(self._targetPos, speed=self._speed, **kwargs)
			spin = vizact.spinTo(euler=self._targetEuler, speed=self._rotationalSpeed, **kwargs)
		
		# Run the actions in separate pools, as needed
		if self._translationMode != DISABLED:
			self.runAction(move, 0)
		if self._rotationMode != DISABLED:
			self.runAction(spin, 1)


if __name__ == "__main__":
	import vizshape
	import vizconfig
	
	viz.go()
	
	# Load an environment model
	piazza = viz.add('piazza.osgb')
	
	# Set up a convenient view
	viz.MainView.setPosition(3, 8, 0.5)
	viz.MainView.setEuler(0, 90, 0)
	
	viz.window.setSize(1600, 1200)
	viz.MainWindow.ortho(-8, 6, -8, 6, 0.1, 100)
	
	# Add a grid
	grid = vizshape.addGrid()
	grid.setPosition(0, 0.1, 0)
	grid.color(viz.BLACK)
	
	# Set up a representation for a transport
	transportRepresentation = vizshape.addAxes()
	transportText = viz.addText('Transport', pos=[0.1, 0.2, -0.1], scale=[0.4] * 3, euler=[0, 90, 0])
	transportText.setBackdrop(viz.BACKDROP_RIGHT_BOTTOM)
	transportText.alignment(viz.ALIGN_LEFT_TOP)
	transportText.setParent(transportRepresentation)
	
	# Set up a representation for a pivot
	pivotRepresentation = vizshape.addAxes()
	pivotText = viz.addText('Pivot', pos=[0.1, 0.2, 0.1], scale=[0.4] * 3, euler=[0, 90, 0])
	pivotText.setBackdrop(viz.BACKDROP_RIGHT_BOTTOM)
	pivotText.setParent(pivotRepresentation)
	pivotRepresentation.setPosition(0, 0, 1)
	pivotRepresentation.setEuler(-20, 0, 0)
	
	# Set the pivot representation as a child of the transport
	pivotRepresentation.setParent(transportRepresentation)
	
	# Instantiate a transport with above choices
	myTeleport = Teleport(node=transportRepresentation)
	
	# Set the pivot representation to be the pivot of the transport
	myTeleport.setPivot(pivotRepresentation)
	
	
	# Define a number of functions to be called by the configurable
	def resetTeleport():
		myTeleport.setPosition(0, 0, 0)
		myTeleport.setEuler(0, 0, 0)
		bc.updateConfigUI()
	
	def startTeleport():
		myTeleport.teleportTo([1, 0, 1], [45, 0, 0])
	
	def getTransportOriOffset():
		return myTeleport.getEuler() != [0, 0, 0]
	
	def setTransportOriOffset(state):
		myTeleport.setPosition(0, 0, 0)  # prettier to zero out here
		if state:
			myTeleport.setEuler(5, 0, 0)
		else:
			myTeleport.setEuler(0, 0, 0)
	
	def getPivotPosOffset():
		return pivotRepresentation.getPosition() != [0, 0, 0]
	
	def setPivotPosOffset(state):
		if state:
			pivotRepresentation.setPosition(0, 0, 1)
		else:
			pivotRepresentation.setPosition(0, 0, 0)
	
	def getPivotOriOffset():
		return pivotRepresentation.getEuler() != [0, 0, 0]
	
	def setPivotOriOffset(state):
		if state:
			pivotRepresentation.setEuler(-20, 0, 0)
		else:
			pivotRepresentation.setEuler(0, 0, 0)
	
	
	# Create a configurable with items to control the demo
	bc = vizconfig.BasicConfigurable('Teleport demo controls')
	
	# Buttons to make it go
	bc.addCommandItem('Reset Transport to initial state', 'Reset', fup=resetTeleport)
	bc.addCommandItem('Teleport to: [1, 0, 1], [45, 0, 0]', 'Teleport', fup=startTeleport)
	
	# Options to add a variety of initial conditions
	bc.addBoolItem('Offset transport ori by: [5, 0, 0]', setTransportOriOffset, getTransportOriOffset)
	bc.addBoolItem('Offset pivot pos by: [0, 0, 1]', setPivotPosOffset, getPivotPosOffset)
	bc.addBoolItem('Offset pivot ori by: [-20, 0, 1]', setPivotOriOffset, getPivotOriOffset)
	
	# Create config window, register the configurable, make it visible
	myWindow = vizconfig.getConfigWindow(name='my_config_window')
	vizconfig.register(bc, window='my_config_window')
	myWindow.setWindowAlignment(vizconfig.ALIGN_LEFT_TOP)
	vizconfig.getConfigWindow('my_config_window').setWindowVisible(True)
	
	# Register the transport to the default configuration window
	vizconfig.register(myTeleport)
	vizconfig.getConfigWindow().setWindowVisible(True)
